<?php

if (!defined('ABSPATH')) exit; // Exit if accessed directly

/**
 * Plugin Name: Gateline Payment Gateway
 * Plugin URI: ---
 * Description: Provides a Gateline Payment Gateway.
 * Version: 1.0.1
 * Author: ---
 */


/* Add a custom payment class to WC
  ------------------------------------------------------------ */
add_action('plugins_loaded', 'woocommerce_gateline', 0);
function woocommerce_gateline()
{
	if (!class_exists('WC_Payment_Gateway'))
		return; // if the WC payment gateway class is not available, do nothing
	if (class_exists('WC_Gateline'))
		return;

	class WC_Gateline extends WC_Payment_Gateway
	{
		public function __construct()
		{

			$plugin_dir = plugin_dir_url(__FILE__);

			global $woocommerce;

			$this->id = 'gateline';
			$this->icon = apply_filters('woocommerce_gateline_icon', '' . $plugin_dir . 'gateline.png');
			$this->has_fields = false;

			// Load settings
			$this->init_form_fields();
			$this->init_settings();

			// Define user set variables
			$this->title = $this->get_option('title');
			$this->MNT_URL = $this->get_option('MNT_URL');
			$this->MNT_ID = $this->get_option('MNT_ID');
			$this->MNT_DATAINTEGRITY_CODE = $this->get_option('MNT_DATAINTEGRITY_CODE');
			// $this->MNT_TEST_MODE = $this->get_option('MNT_TEST_MODE');
			// $this->debug = $this->get_option('debug');
			$this->description = $this->get_option('description');
			$this->instructions = $this->get_option('instructions');

			// Actions
			add_action('woocommerce_receipt_gateline', array($this, 'receipt_page'));

			// Save options
			add_action('woocommerce_update_options_payment_gateways_gateline', array($this, 'process_admin_options'));

			// Payment listener/API hook
			add_action('woocommerce_api_wc_gateline', array($this, 'check_assistant_response'));

			if (!$this->is_valid_for_use()) {
				$this->enabled = false;
			}
		}

		/**
		 * Check if this gateway is enabled and available in the user's country
		 */
		function is_valid_for_use()
		{
			if (!in_array(get_option('woocommerce_currency'), array('RUB'))) {
				return false;
			}

			return true;
		}

		/**
		 * Admin Panel Options
		 * - Options for bits like 'title' and availability on a country-by-country basis
		 *
		 * @since 0.1
		 **/
		public function admin_options()
		{
			?>
			<h3><?php _e('Gateline', 'woocommerce'); ?></h3>
			<p><?php _e('Настройка приема электронных платежей через Gateline.', 'woocommerce'); ?></p>

			<?php if ($this->is_valid_for_use()) : ?>

				<table class="form-table">

					<?php
				// Generate the HTML For the settings form.
					$this->generate_settings_html();
					?>
				</table><!--/.form-table-->

			<?php else : ?>
				<div class="inline error"><p>
					<strong><?php _e('Шлюз отключен', 'woocommerce'); ?></strong>: <?php _e('Gateline не поддерживает валюты Вашего магазина.', 'woocommerce'); ?>
				</p></div>
				<?php
				endif;

		} // End admin_options()

		/**
		 * Initialise Gateway Settings Form Fields
		 *
		 * @access public
		 * @return void
		 */
		function init_form_fields()
		{
			$this->form_fields = array(
				'enabled'                => array(
					'title'   => __('Включить/Выключить', 'woocommerce'),
					'type'    => 'checkbox',
					'label'   => __('Включен', 'woocommerce'),
					'default' => 'yes'
					),
				'title'                  => array(
					'title'       => __('Название', 'woocommerce'),
					'type'        => 'text',
					'description' => __('Это название, которое пользователь видит во время проверки.', 'woocommerce'),
					'default'     => __('Gateline', 'woocommerce')
					),
				'MNT_ID'                 => array(
					'title'       => __('Логин', 'woocommerce'),
					'type'        => 'text',
					'description' => __('Пожалуйста введите UID сайта.<br/><b>Внимание!</b>UID сайта отличается для тестового и рабочего аккаунтов.', 'woocommerce'),
					'default'     => '99999999'
					),
				'MNT_DATAINTEGRITY_CODE' => array(
					'title'       => __('Пароль сайта', 'woocommerce'),
					'type'        => 'text',
					'description' => __('Пожалуйста введите Пароль сайта', 'woocommerce'),
					'default'     => '******'
					),
				'MNT_URL'                => array(
					'title'       => __('Режим оплаты', 'woocommerce'),
					'type'        => 'select',
					'options'     => array(
						'simpleapi.sandbox.gateline.net:18610'   => 'Тестовый (simpleapi.sandbox.gateline.net:18610)',
						'simpleapi.gateline.ru' => 'Боевой (simpleapi.gateline.ru)'
						),
					'description' => __('Пожалуйста выберите режим оплаты.', 'woocommerce'),
					'default'     => ''
					),
				// 'MNT_TEST_MODE'          => array(
				// 	'title'       => __('Тестовый режим', 'woocommerce'),
				// 	'type'        => 'checkbox',
				// 	'label'       => __('Включен', 'woocommerce'),
				// 	'description' => __('В этом режиме плата за товар не снимается.', 'woocommerce'),
				// 	'default'     => 'no'
				// 	),
				// 'debug'                  => array(
				// 	'title'   => __('Debug', 'woocommerce'),
				// 	'type'    => 'checkbox',
				// 	'label'   => __('Включить логирование (<code>woocommerce/logs/gateline.txt</code>)', 'woocommerce'),
				// 	'default' => 'no'
				// 	),
				'description'            => array(
					'title'       => __('Description', 'woocommerce'),
					'type'        => 'textarea',
					'description' => __('Описанием метода оплаты которое клиент будет видеть на вашем сайте.', 'woocommerce'),
					'default'     => 'Оплата с использованием банковской карты (Gateline).'
					),
				'instructions'           => array(
					'title'       => __('Instructions', 'woocommerce'),
					'type'        => 'textarea',
					'description' => __('Инструкции, которые будут добавлены на страницу благодарностей.', 'woocommerce'),
					'default'     => 'Оплата с использованием банковской карты (Gateline).'
					)
				);
		}

		/**
		 * Дополнительная информация в форме выбора способа оплаты
		 **/
		function payment_fields()
		{
			if ($this->description) {
				echo wpautop(wptexturize($this->description));
			}
		}

		/**
		 * Process the payment and return the result
		 **/
		function process_payment($order_id)
		{
			$order = new WC_Order($order_id);

			return array(
				'result'   => 'success',
				'redirect' => add_query_arg('order', $order->id, add_query_arg('key', $order->order_key, get_permalink(woocommerce_get_page_id('pay'))))
				);
		}

		/**
		 * Форма оплаты
		 **/
		function receipt_page($order_id)
		{
			echo '<p>' . __('Спасибо за Ваш заказ, пожалуйста, нажмите кнопку ниже, чтобы заплатить.', 'woocommerce') . '</p>';

			$order = new WC_Order($order_id);

			$amount = number_format($order->order_total, 2, '.', '');
			// $test_mode = ($this->MNT_TEST_MODE == 'yes') ? 1 : 0;
			$currency = get_woocommerce_currency();
			if ($currency == 'RUR') $currency = 'RUB';
			$description = "Оплата заказа #".$order_id;
			$email = $order->billing_email;
			$message = "amount=".$amount.";description=".$description.";email=".$email.";merchant_order_id=".$order_id.";site=".$this->MNT_ID;
			$signature = hash_hmac('sha1', $message, $this->MNT_DATAINTEGRITY_CODE);

			$args = array(
				'site'             => $this->MNT_ID,
				'amount'         => $amount,
				'email'          => $email,
				'description'         => $description,
				'merchant_order_id' => $order_id,
				'checksum'      => $signature
				// 'MNT_TEST_MODE'      => $test_mode,
				// 'MNT_CURRENCY_CODE'  => $currency,
				);

			$args_array = array();

			foreach ($args as $key => $value) {
				$args_array[] = '<input type="hidden" name="' . esc_attr($key) . '" value="' . esc_attr($value) . '" />';
			}

			echo '<form action="' . esc_url("https://" . $this->MNT_URL . "/pay") . '" method="POST" id="gateline_payment_form">' . "\n" .
			implode("\n", $args_array) .
			'<input type="submit" class="button alt" id="submit_gateline_payment_form" value="' . __('Оплатить', 'woocommerce') . '" /> <a class="button cancel" href="' . $order->get_cancel_order_url() . '">' . __('Отказаться от оплаты и вернуться в корзину', 'woocommerce') . '</a>' . "\n" .
			'</form>';
		}

		/**
		 * Check URL validity
		 **/
		function check_assistant_request_is_valid($posted)
		{
			if (isset($posted['gateline']) && isset($posted['status']) && isset($posted['wc-api']) && isset($posted['message'])) {
				$message = "";
				if (isset($posted['code'])) { $message .= "code=".$posted['code'].";"; }
				$message .= "gateline=".$posted['gateline'].";";
				if (isset($posted['merchant_order_id'])) { $message .= "merchant_order_id=".$posted['merchant_order_id'].";"; }
				$message .= "message=".$posted['message'].";";
				if (isset($posted['order_id'])) { $message .= "order_id=".$posted['order_id'].";"; }
				$message .= "status=".$posted['status'].";";
				$message .= "wc-api=".$posted['wc-api'];
				$signature = hash_hmac('sha1', $message, $this->MNT_DATAINTEGRITY_CODE);
				// die("checksum: ".$posted['checksum']."<br/>"."signature: ".$signature."<br/>"."message: ".$message);
				if ($posted['checksum'] !== $signature) {
					return false;
				}
			} else {
				return false;
			}

			return true;
		}

		/**
		 * Check Response
		 **/
		function check_assistant_response()
		{
			global $woocommerce;

			$_REQUEST = stripslashes_deep($_REQUEST);
			if (isset($_REQUEST['gateline']) AND $_REQUEST['gateline'] == 'callback') {
				@ob_clean();

				if ($this->check_assistant_request_is_valid($_REQUEST)) {
					$order = new WC_Order($_REQUEST['merchant_order_id']);

					// Check order not already completed
					if ($order->status == 'completed') {
						die('FAIL');
					}

					// Payment completed
					$order->add_order_note(__('Платеж успешно завершен.', 'woocommerce'));
					$order->payment_complete();

					die('SUCCESS');
				} else {
					die('FAIL');
				}
			} else if (isset($_REQUEST['gateline']) AND $_REQUEST['gateline'] == 'success') {
				if ($this->check_assistant_request_is_valid($_REQUEST)) {
					$order = new WC_Order($_REQUEST['merchant_order_id']);
					$order->update_status('on-hold', __('Заказ успешно оплачен', 'woocommerce'));
					$woocommerce->cart->empty_cart();

					wp_redirect($this->get_return_url($order));
					exit;
				} else {
					die('Not correct request');
				}
			} else if (isset($_REQUEST['gateline']) AND $_REQUEST['gateline'] == 'fail') {
				if ($this->check_assistant_request_is_valid($_REQUEST)) {
					$order = new WC_Order($_REQUEST['merchant_order_id']);
					$order->update_status('failed', __('Заказ не оплачен', 'woocommerce'));

					wp_redirect($order->get_cancel_order_url());
					exit;
				} else {
					die('Not correct request');
				}
			}

		}

	}

	/**
	 * Add the gateway to WooCommerce
	 **/
	function add_gateline_gateway($methods)
	{
		$methods[] = 'WC_Gateline';

		return $methods;
	}

	add_filter('woocommerce_payment_gateways', 'add_gateline_gateway');
}

?>
